#!/bin/sh
#
# Copyright 2017 Google Inc. All Rights Reserved.
#

# <cloud-sdk-sh-preamble>
#
#  CLOUDSDK_ROOT_DIR            (a)  installation root dir
#  CLOUDSDK_PYTHON              (u)  python interpreter path
#  CLOUDSDK_GSUTIL_PYTHON       (u)  python interpreter path for gsutil
#  CLOUDSDK_PYTHON_ARGS         (u)  python interpreter arguments
#  CLOUDSDK_PYTHON_SITEPACKAGES (u)  use python site packages
#
# (a) always defined by the preamble
# (u) user definition overrides preamble

# Wrapper around 'which' and 'command -v', tries which first, then falls back
# to command -v
_cloudsdk_which() {
  which "$1" 2>/dev/null || command -v "$1" 2>/dev/null
}

# Check whether passed in python command reports major version 3.
_is_python3() {
  echo "$("$1" -V 2>&1)" | grep -E "Python 3" > /dev/null
}

# For Python 3, gsutil requires Python 3.5+.
_py3_interpreter_compat_with_gsutil () {
  # Some environments (e.g. macOS) don't support grep -P, so we use grep -E.
  echo "$("$1" -V 2>&1)" | grep -E "Python 3[.]([5-9]|[1-9][0-9])" > /dev/null
}

# Determines the real cloud sdk root dir given the script path.
# Would be easier with a portable "readlink -f".
_cloudsdk_root_dir() {
  case $1 in
  /*)   _cloudsdk_path=$1
        ;;
  */*)  _cloudsdk_path=$PWD/$1
        ;;
  *)    _cloudsdk_path=$(_cloudsdk_which $1)
        case $_cloudsdk_path in
        /*) ;;
        *)  _cloudsdk_path=$PWD/$_cloudsdk_path ;;
        esac
        ;;
  esac
  _cloudsdk_dir=0
  while :
  do
    while _cloudsdk_link=$(readlink "$_cloudsdk_path")
    do
      case $_cloudsdk_link in
      /*) _cloudsdk_path=$_cloudsdk_link ;;
      *)  _cloudsdk_path=$(dirname "$_cloudsdk_path")/$_cloudsdk_link ;;
      esac
    done
    case $_cloudsdk_dir in
    1)  break ;;
    esac
    if [ -d "${_cloudsdk_path}" ]; then
      break
    fi
    _cloudsdk_dir=1
    _cloudsdk_path=$(dirname "$_cloudsdk_path")
  done
  while :
  do  case $_cloudsdk_path in
      */)     _cloudsdk_path=$(dirname "$_cloudsdk_path/.")
              ;;
      */.)    _cloudsdk_path=$(dirname "$_cloudsdk_path")
              ;;
      */bin)  dirname "$_cloudsdk_path"
              break
              ;;
      *)      echo "$_cloudsdk_path"
              break
              ;;
      esac
  done
}
CLOUDSDK_ROOT_DIR=$(_cloudsdk_root_dir "$0")

setup_cloudsdk_python() {
  if [ -z "$CLOUDSDK_PYTHON" ]; then
    # if python2 exists then plain python may point to a version != 2
    if _cloudsdk_which python2 >/dev/null; then
      CLOUDSDK_PYTHON=python2
    elif _cloudsdk_which python2.7 >/dev/null; then
      # this is what some OS X versions call their built-in Python
      CLOUDSDK_PYTHON=python2.7
    elif _cloudsdk_which python >/dev/null; then
      # Use unversioned python if it exists.
      CLOUDSDK_PYTHON=python
    elif _cloudsdk_which python3 >/dev/null; then
      # We support python3, but only want to default to it if nothing else is
      # found.
      CLOUDSDK_PYTHON=python3
    else
      # This won't work because it wasn't found above, but at this point this
      # is our best guess for the error message.
      CLOUDSDK_PYTHON=python
    fi
  fi
}

setup_cloudsdk_python_prefer_python3() {
# Settings for corp
  if [ -z "$CLOUDSDK_PYTHON" ]; then
    # if python3 exists then plain python may point to a version != 3
    if _cloudsdk_which python3 >/dev/null; then
      CLOUDSDK_PYTHON=python3
    elif _cloudsdk_which python2.7 >/dev/null; then
      # this is what some OS X versions call their built-in Python
      CLOUDSDK_PYTHON=python2.7
    elif _cloudsdk_which python >/dev/null; then
      # Use unversioned python if it exists.
      CLOUDSDK_PYTHON=python
    elif _cloudsdk_which python2 >/dev/null; then
      # We support python2, but only want to default to it if nothing else is
      # found.
      CLOUDSDK_PYTHON=python2
    else
      # This won't work because it wasn't found above, but at this point this
      # is our best guess for the error message.
      CLOUDSDK_PYTHON=python
    fi
  fi
}

case $HOSTNAME in
  *.corp.google.com) setup_cloudsdk_python_prefer_python3;;
  *.c.googlers.com) setup_cloudsdk_python_prefer_python3;;
  *) setup_cloudsdk_python;;
esac

# $PYTHONHOME can interfere with gcloud. Users should use
# CLOUDSDK_PYTHON to configure which python gcloud uses.
unset PYTHONHOME

# if CLOUDSDK_PYTHON_SITEPACKAGES and VIRTUAL_ENV are empty
case :$CLOUDSDK_PYTHON_SITEPACKAGES:$VIRTUAL_ENV: in
:::)  # add -S to CLOUDSDK_PYTHON_ARGS if not already there
      case " $CLOUDSDK_PYTHON_ARGS " in
      *" -S "*) ;;
      "  ")     CLOUDSDK_PYTHON_ARGS="-S"
                ;;
      *)        CLOUDSDK_PYTHON_ARGS="$CLOUDSDK_PYTHON_ARGS -S"
                ;;
      esac
      unset CLOUDSDK_PYTHON_SITEPACKAGES
      ;;
*)    # remove -S from CLOUDSDK_PYTHON_ARGS if already there
      while :; do
        case " $CLOUDSDK_PYTHON_ARGS " in
        *" -S "*) CLOUDSDK_PYTHON_ARGS=${CLOUDSDK_PYTHON_ARGS%%-S*}' '${CLOUDSDK_PYTHON_ARGS#*-S} ;;
        *) break ;;
        esac
      done
      # if CLOUDSDK_PYTHON_SITEPACKAGES is empty
      [ -z "$CLOUDSDK_PYTHON_SITEPACKAGES" ] &&
        CLOUDSDK_PYTHON_SITEPACKAGES=1
      export CLOUDSDK_PYTHON_SITEPACKAGES
      ;;
esac

# TODO(b/133246173): Remove this once we want to default to Python 3.
# Allow users to set the Python interpreter used to launch gsutil, falling
# back to the CLOUDSDK_PYTHON interpreter otherwise. In the future, if this
# is not set, we'll try finding (and prefer using) Python 3 before falling
# back to the default Cloud SDK Python.
if [ -z "$CLOUDSDK_GSUTIL_PYTHON" ]; then
  CLOUDSDK_GSUTIL_PYTHON="$CLOUDSDK_PYTHON"
fi

# Gsutil prefers Python 3 if it is available, which may likely differ from the
# $CLOUDSDK_PYTHON version. We launch Gsutil with $CLOUDSDK_GSUTIL_PYTHON; the
# user can set this to any interpreter they like, so we only try to find
# Python 3 for them if they haven't specified the interpreter already.
if [ -z "$CLOUDSDK_GSUTIL_PYTHON" ]; then
  if _cloudsdk_which python3 >/dev/null && \
       _py3_interpreter_compat_with_gsutil python3; then
    # Try `python3` first.
    CLOUDSDK_GSUTIL_PYTHON=python3
  elif _cloudsdk_which python >/dev/null && \
      _py3_interpreter_compat_with_gsutil python; then
    # If `python3` isn't found or valid, try `python`.
    CLOUDSDK_GSUTIL_PYTHON=python
  else
    # Python 3 doesn't appear to be in the OS path. Use $CLOUDSDK_PYTHON.
    CLOUDSDK_GSUTIL_PYTHON="$CLOUDSDK_PYTHON"
  fi
fi

# bq requires python2 so it can not use CLOUDSDK_PYTHON which may be python3.
if [ -z "$CLOUDSDK_BQ_PYTHON" ]; then
  # if python2 exists then plain python may point to a version != 2
  if _cloudsdk_which python2 >/dev/null; then
    CLOUDSDK_BQ_PYTHON=python2
  elif _cloudsdk_which python2.7 >/dev/null; then
    # this is what some OS X versions call their built-in Python
    CLOUDSDK_BQ_PYTHON=python2.7
  elif _cloudsdk_which python >/dev/null; then
    # Use unversioned python if it exists.
    CLOUDSDK_BQ_PYTHON=python
  else
    # This won't work because it wasn't found above, but at this point this
    # is our best guess for the error message.
    CLOUDSDK_BQ_PYTHON=python
  fi
fi

export CLOUDSDK_ROOT_DIR
export CLOUDSDK_PYTHON_ARGS
export CLOUDSDK_GSUTIL_PYTHON
export CLOUDSDK_BQ_PYTHON

# </cloud-sdk-sh-preamble>

exec "$CLOUDSDK_PYTHON" $CLOUDSDK_PYTHON_ARGS "${CLOUDSDK_ROOT_DIR}/bin/bootstrapping/java_dev_appserver.py" "$@"
